<?php
// file.php

// Some standard functions for working with files and discovering files.
// --

// Tail-recursive form of ac_file_find.  If a file is encountered which
// is a directory, but not the magic . and .. directories, then
// ac_file_find_r recurses into those directories and looks for files in
// them.  All files returned in the $files array use paths derived from
// $base.  If you use '.' in $base then the paths will be based on
// that; if you use dirname(...), the path will be absolute.

function ac_file_find_r($base, $patn, &$files) {
		if (!is_dir($base))
        return array();

    $dir = @opendir($base);

    while (($file = @readdir($dir)) !== false) {
        if (substr($file, 0, 1) != ".") {
	        if (is_dir($base . DIRECTORY_SEPARATOR . $file))
	            ac_file_find_r($base . DIRECTORY_SEPARATOR . $file, $patn, $files);
	        elseif ($patn == '' || preg_match('/'.$patn.'/', $file))
	            $files[] = $base . DIRECTORY_SEPARATOR . $file;
        }
    }

		@closedir($dir);
    return $files;
}

// Given $base, a string path representing where to begin looking for
// files, and $patn, a regular expression describing the filenames you
// are looking for, return an array list of all matching files from
// $base on down.

function ac_file_find($base, $patn = '') {
	$files = array();
	return ac_file_find_r($base, $patn, $files);
}

// List the directories in a given directory. $base should point to the directory that you want to look in.
// $base could be something like: ac_base("templates"), which points us to \knowledgebuilder\templates\
function ac_dir_list($base) {
	$files = array();
	return ac_dir_list_r($base, $files);
}

function ac_dir_list_r($base, &$files) {
	if (!is_dir($base)) return;

    $dir = @opendir($base);

    while (($file = @readdir($dir)) !== false) {
        if (substr($file, 0, 1) != ".") {
	        if (is_dir($base . DIRECTORY_SEPARATOR . $file))
	            $files[] = $base . DIRECTORY_SEPARATOR . $file;
        }
    }

	@closedir($dir);
	return $files;
}

// Alias for file_get_contents().

function ac_file_get($fname) {
    return file_get_contents($fname);
}

function ac_file_get_lines($fname) {
    $fd    = @fopen($fname, "r");
    $lines = array();

    if ($fd == false)
        return $lines;

    while (!feof($fd)) {
        $line = fgets($fd);

        if ($line == false)
            return $lines;

        $lines[] = $line;
    }

    return $lines;
}

// Replacement for file_put_contents(), which is not in PHP prior to
// version 5.

function ac_file_put($fname, $str) {
    $fd = @fopen($fname, "w");

    if ($fd == false)
        return false;

    fwrite($fd, $str);
    fclose($fd);

    return true;
}

function ac_file_put_lines($fname, &$lines) {
    $fd = @fopen($fname, "w");

    if ($fd == false)
        return false;

    foreach ($lines as $line)
        fwrite($fd, $line."\n");

    fclose($fd);
    return true;
}

function ac_file_copy_recursive($source, $dest) {
	// Simple copy for a file
	if ( is_file($source) ) {
		return copy($source, $dest);
	}

	// Make destination directory
	if (!is_dir($dest)) {
		mkdir($dest, 0755);
	}

	// Loop through the folder
	$dir = dir($source);
	while ( false !== $entry = $dir->read() ) {
		// Skip pointers
		if ( $entry == '.' || $entry == '..' ) {
			continue;
		}

		// Deep copy directories
		if ( $dest !== $source . '/' . $entry ) {
			ac_file_copy_recursive($source . '/' . $entry, $dest . '/' . $entry);
		}
	}

	// Clean up
	$dir->close();
	return true;
}

function ac_file_readline($fp) {
	$chunklen = 128;
	$out      = array();
	$fileoff  = @ftell($fp);
	$str      = "";

	while (!feof($fp)) {
		$str   .= @fread($fp, $chunklen);
		$lfoff = strpos($str, "\n");
		$croff = strpos($str, "\r");

		if ($lfoff === false && $croff === false) {
			if (feof($fp)) {
				$out[] = $str;
				break;
			}
			continue;
		}

		if ($lfoff !== false) {
			@fseek($fp, $fileoff + $lfoff + 1);
			$out[] = substr($str, 0, $lfoff);
			break;
		}

		if ($lfoff === false) {		# $croff must not be false at this point.
			@fseek($fp, $fileoff + $croff + 1);
			$out[] = substr($str, 0, $croff);
			break;
		}

		$out[] = $str;
		$str   = "";
	}

	return implode("", $out);
}

function ac_file_rmdir_recursive($dirname) {
	// Sanity check
	if ( !file_exists($dirname) ) {
		return false;
	}

	// Simple delete for a file
	if ( is_file($dirname) ) {
		return unlink($dirname);
	}

	// Loop through the folder
	$dir = dir($dirname);
	while ( false !== $entry = $dir->read() ) {
		// Skip pointers
		if ( $entry == '.' || $entry == '..' ) {
			continue;
		}

		// Recurse
		ac_file_rmdir_recursive($dirname . DIRECTORY_SEPARATOR . $entry);
	}

	// Clean up
	$dir->close();
	return rmdir($dirname);
}


function ac_file_basename($path) {
	$path = basename($path);
	return (strlen($path) > 0 && substr($path, 0, 1) != '.' ? $path : '') ;
}


function ac_file_humansize($size) {
	$count = 0;
	$format = array('B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB');
	while( ( $size / 1024 ) > 1 and $count < 8 ) {
		$size = $size / 1024;
		$count++;
	}
	$decimals = (int)( $size < 10 );
	return number_format($size, $decimals, '.', ' ') . ' ' . $format[$count];
}



function ac_file_upload($prefix, $tablef, $tabled, $relid, $relval, $maxsize = null) {
	if ( is_null($maxsize) ) {
		$site = ac_site_get();
		$maxsize = (int)$site['maxuploadfilesize'];
	}
	$r = array(
		'succeeded' => false,
		'message' => _a('Command not provided.'),
		'id' => 0,
		'filename' => '',
		'filesize' => 0,
		'relid' => $relval
	);
	// require PEAR package for file uploads
	require_once(ac_global('pear/Upload.php'));
	// if requested upload, upload it
	$upload = new HTTP_Upload('en');
	// Set allowed file extension
	$file = $upload->getFiles('ac_uploader');
	$file->setValidExtensions(array('php'), 'deny');
	if ( PEAR::isError($file) || !$file->isValid() ) {
		$r['message'] = $file->getMessage();
		if ( isset($_FILES['ac_uploader']['name']) ) $r['filename'] = $_FILES['ac_uploader']['name'];
		return $r;
	}
	// Get a "safe" file name
	$r['filename'] = preg_replace('/\.php/i', '', $file->setName('real'));
	$file->setName($r['filename']);
	// Check the file size
	$r['filesize'] = $size = $file->getProp('size');
	if  ( $maxsize > 0 and $r['filesize'] > $maxsize * 1024 * 1024 ) { // this is in megs
		$r['message'] = _a("The file is too big! Maximum allowed filesize is %sMB.", $maxsize);
		return $r;
	}
	// upload a file and return a message
	$realpath = $file->upload['tmp_name'];
	$type = ( strlen($tabled) > 0 && substr($tabled, 0, 1) == '#' ? 'db' : 'fs' ); // if database table provided, save 2 db, otherwise 2 filesystem
	$reltype = ( strlen($tablef) > 0 && substr($tablef, 0, 1) == '#' ? 'db' : 'fs' ); // if database table provided, save 2 db, otherwise 2 filesystem
	// Insert it
	if ( $reltype == 'fs' ) {
		// save to filesystem
		$fn = $prefix . '-' . $r['filename'];
		$file->setName($fn);
		$r['succeeded'] = $file->moveTo($tablef, true);
		$r['id'] = $fileID = $fn;
		if ( !$r['succeeded'] ) {
			$r['message'] = $file->getMessage();
			return $r;
		}
	} else {
		// save file info in database
		$insert = array(
			'id' => 0,
			$relid => $relval,
			'name' => $r['filename'],
			'size' => $r['filesize'],
			'mime_type' => $file->getProp('type'),
			'=tstamp' => 'NOW()'
		);
		if ( ac_ihook_exists('ac_file_upload_info') ) {
			$insert = ac_ihook('ac_file_upload_info', $file, $insert);
		}
		$sql = ac_sql_insert($tablef, $insert);
		if ( !$sql ) {
			$r['message'] = ac_sql_error();
			return $r;
		}
		$r['id'] = $fileID = (int)ac_sql_insert_id();
	}
	if ( $type == 'fs' ) {
		if ( $reltype != 'fs' ) { // if this is also fs, it's saved already
			// save to filesystem
			$file->setName($prefix . '-' . $fileID);
			$r['succeeded'] = $file->moveTo($tabled, true);
		} else {
			$r['succeeded'] = true;
		}
		if ( !$r['succeeded'] ) {
			$r['message'] = $file->getMessage();
			return $r;
		}
	} else {
		// save file content in database
		// Place holder
		$currentPos = 0;
		// Loop counter
		$count = 1;
		// Chunk size
		$chunkSize = 700000;
		// insert array
		$insert = array(
			'id' => 0,
			'fileid' => $fileID,
			'sequence' => 1,
			'data' => ''
		);
		// Get data
		$data = ac_file_get($realpath); // we can do fopen/fseek/fread combo for optimization
		$size = strlen($data);
		// Loop
		while ( $currentPos < $size ) {
			// Get a order number
			$insert['sequence'] = $count;
			// Get a chunk of the data
			$insert['data'] = substr($data, $currentPos, $chunkSize);
			// Insert it
			$retval = ac_sql_insert($tabled, $insert);
			if ( !$retval ) {
				$r['message'] = ac_sql_error();
				$r['succeeded'] = false;
				// If this is ever false we should remove everything about this file from
				// the database.
				ac_sql_query("DELETE FROM `$tablef` WHERE `id` = '$fileID'");
				ac_sql_query("DELETE FROM `$tabled` WHERE `fileid` = '$fileID'");
				return $r;
			}
			// Update the current position
			$currentPos += $chunkSize;
			$count++;
		}
		$r['succeeded'] = true;
	}
	if ( $r['succeeded'] ) {
		$r['message'] = _a('File %s uploaded.', $r['filename']);
	} else {
		$r['message'] = _a('Upload failed for file %s.', $r['filename']);
	}
	return $r;
}

function ac_file_save($prefix, $tablef, $tabled, $relid, $relval, $maxsize = null, $file) {
	if ( is_null($maxsize) ) {
		$site = ac_site_get();
		$maxsize = (int)$site['maxuploadfilesize'];
	}
	$r = array(
		'succeeded' => false,
		'message' => _a('File not provided.'),
		'id' => 0,
		'filename' => '',
		'filesize' => '',
		'relid' => $relval
	);
	if ( !isset($file['name']) or !isset($file['size']) or !isset($file['data']) ) {
		return $r;
	}
	// Get a "safe" file name
	$r['filename'] = $file['name'] = ac_str_urlsafe($file['name']);
	// Check the file size
	$r['filesize'] = $size = $file['size'];
	if  ( $maxsize > 0 and $r['filesize'] > $maxsize * 1024 * 1024 ) { // this is in megs
		$r['message'] = _a("The file is too big! Maximum allowed filesize is %sMB.", $maxsize);
		return $r;
	}
	// save a file and return a message
	$type = ( strlen($tabled) > 0 && substr($tabled, 0, 1) == '#' ? 'db' : 'fs' ); // if database table provided, save 2 db, otherwise 2 filesystem
	$reltype = ( strlen($tablef) > 0 && substr($tablef, 0, 1) == '#' ? 'db' : 'fs' ); // if database table provided, save 2 db, otherwise 2 filesystem
	// Insert it
	if ( $reltype == 'fs' ) {
		// save to filesystem
		$fn = $prefix . '-' . $r['filename'];
		$r['succeeded'] = ac_file_put($tablef, $file['data']);
		$r['id'] = $fileID = $fn;
		if ( !$r['succeeded'] ) {
			$r['message'] = _a('File info could not be saved');
			return $r;
		}
	} else {
		// save file info in database
		$insert = array(
			'id' => 0,
			$relid => $relval,
			'name' => $r['filename'],
			'size' => $r['filesize'],
			'mime_type' => $file['mimetype'],
			'=tstamp' => 'NOW()'
		);
		$sql = ac_sql_insert($tablef, $insert);
		if ( !$sql ) {
			$r['message'] = ac_sql_error();
			return $r;
		}
		$r['id'] = $fileID = (int)ac_sql_insert_id();
	}
	if ( $type == 'fs' ) {
		if ( $reltype != 'fs' ) { // if this is also fs, it's saved already
			// save to filesystem
			$fn = $prefix . '-' . $fileID;
			$r['succeeded'] = ac_file_put($tabled, $file['data']);
		} else {
			$r['succeeded'] = true;
		}
		if ( !$r['succeeded'] ) {
			$r['message'] = _a('File data could not be saved');
			return $r;
		}
	} else {
		// save file content in database
		// Place holder
		$currentPos = 0;
		// Loop counter
		$count = 1;
		// Chunk size
		$chunkSize = 700000;
		// insert array
		$insert = array(
			'id' => 0,
			'fileid' => $fileID,
			'sequence' => 1,
			'data' => ''
		);
		// Get data
		$size = strlen($file['data']);
		// Loop
		while ( $currentPos < $size ) {
			// Get a order number
			$insert['sequence'] = $count;
			// Get a chunk of the data
			$insert['data'] = substr($file['data'], $currentPos, $chunkSize);
			// Insert it
			$retval = ac_sql_insert($tabled, $insert);
			if ( !$retval ) {
				$r['message'] = ac_sql_error();
				$r['succeeded'] = false;
				// If this is ever false we should remove everything about this file from
				// the database.
				ac_sql_query("DELETE FROM `$tablef` WHERE `id` = '$fileID'");
				ac_sql_query("DELETE FROM `$tabled` WHERE `fileid` = '$fileID'");
				return $r;
			}
			// Update the current position
			$currentPos += $chunkSize;
			$count++;
		}
		$r['succeeded'] = true;
	}
	if ( $r['succeeded'] ) {
		$r['message'] = _a('File %s uploaded.', $r['filename']);
	} else {
		$r['message'] = _a('Upload failed for file %s.', $r['filename']);
	}
	return $r;
}

function ac_file_upload_remove($tablef, $tabled, $id) {
	$type = ( strlen($tabled) > 0 && substr($tabled, 0, 1) == '#' ? 'db' : 'fs' ); // if database table provided, save 2 db, otherwise 2 filesystem
	$reltype = ( strlen($tablef) > 0 && substr($tablef, 0, 1) == '#' ? 'db' : 'fs' ); // if database table provided, save 2 db, otherwise 2 filesystem
	// do actual deleting
	if ( $reltype == 'fs' ) {
		return @unlink($tablef . DIRECTORY_SEPARATOR . $id);
	} else {
		return (
			ac_sql_query("DELETE FROM $tablef WHERE `id` = '$id'")
		and
			( $type == 'fs' ? @unlink($tabled . DIRECTORY_SEPARATOR . $id) : ac_sql_query("DELETE FROM $tabled WHERE `fileid` = '$id'") )
		);
	}
}

function ac_file_upload_list($tablef, $relid, $relval, $relop = '=') {
	$site = ac_site_get();
	$r = array();
	$query = "
		SELECT
			*
		FROM
			$tablef f
		WHERE
			`$relid` $relop $relval
		ORDER BY
			`tstamp` ASC
	";
	$sql = ac_sql_query($query);
	if ( !$sql or mysql_num_rows($sql) == 0 ) return $r;
	while ( $row = ac_sql_fetch_assoc($sql, array("tstamp"))) {
		$row['datetime'] = ac_date_format($row['tstamp'], $site['datetimeformat']);
		$row['humansize'] = ac_file_humansize($row['size']);
		$row['hash'] = md5($row[$relid] . '*|*' . $row['name']);
		$row['filetype'] = ac_file_upload_type($row);
		$row['viewable'] = ac_file_upload_viewable($row['filetype']);
		$r[$row['id']] = $row;
	}
	return $r;
}

function ac_file_upload_get($tablef, $tabled, $relid, $id) {
//function ac_file_upload_get($id) {
	$site = ac_site_get();
	$r = false;
	$query = "
		SELECT
			*
		FROM
			`$tablef` f
		WHERE
			`id` = '$id'
	";
	$sql = ac_sql_query($query);
	if ( !$sql or mysql_num_rows($sql) == 0 ) return $r;
	$r = ac_sql_fetch_assoc($sql, array("tstamp"));
	$r['datetime'] = ac_date_format($r['tstamp'], $site['datetimeformat']);
	$r['humansize'] = ac_file_humansize($r['size']);
	$r['hash'] = md5($r[$relid] . '*|*' . $r['name']);
	// get data
	$r['data'] = ac_file_upload_get_data($tabled, $id);
	return $r;
}

function ac_file_upload_get_data($tabled, $id) {
	// if filesystem is used?
	if ( substr($tabled, 0, 1) != '#' ) {
		return ( file_exists($tabled . $id) ? ac_file_get($tabled . $id) : '' );
	} else {
		return implode('', ac_sql_select_list("SELECT `data` FROM `$tabled` WHERE `fileid` = '$id' ORDER BY `sequence`"));
	}
	return '';
}

function ac_file_upload_type($file) {
	// check mime type
	$mime = strtolower($file['mime_type']);
	$type = explode('/', $mime);
	if ( in_array($type[0], array('image', 'text')) ) {
		if ( $type[0] == 'text' and $type[1] == 'html' ) return 'html';
		return $type[0];
	}
	// check extension
	$ext = ( ac_str_instr('.', $file['name']) ? strtolower(end(explode('.', $file['name']))) : '' );
	if ( ac_str_instr('htm', $ext) ) {
		return 'html';
	}
	if ( $ext == 'pdf' or $mime == 'application/pdf' ) return 'pdf';
	return '';
}

function ac_file_upload_viewable($filetype) {
	return in_array($filetype, array('html', 'text', 'image', 'pdf', /*'document'*/));
}

function ac_file_zip_read_xml($filename, $ext, $return_format = "text") {

	// reads the necessary XML file contained within a compressed filetype, such as docx, pptx, odt
	// taken and modified from: http://www.webcheatsheet.com/PHP/reading_the_clean_text_from_docx_odt.php

	if ( class_exists("ZipArchive") ) {

		$zip = new ZipArchive;
		$file_content = "";

		// Open received archive file
		if ($zip -> open($filename) === true) {

			switch ($ext) {

				case "odt" :
					$data_location = "content.xml";
				case "docx" :
					if ( !isset($data_location) ) $data_location = "word/document.xml";
				case "xlsx" :
					if ( !isset($data_location) ) $data_location = "xl/sharedStrings.xml";

					if ( ($index = $zip -> locateName($data_location)) !== false ) {
						$data = $zip -> getFromIndex($index);
						$xml = DOMDocument::loadXML($data, LIBXML_NOENT | LIBXML_XINCLUDE | LIBXML_NOERROR | LIBXML_NOWARNING);
						// return xml by default
						$file_content = $xml -> saveXML();
						if ($return_format == "text") {
							$file_content = strip_tags($file_content);
						}
						elseif ($return_format == "csv") {
							// just process sheet1 for now - later we can add support for every sheet
							$sheet1 = $zip -> locateName("xl/worksheets/sheet1.xml");
							$sheet1 = $zip -> getFromIndex($sheet1);
							$sheet1_xml = DOMDocument::loadXML($sheet1, LIBXML_NOENT | LIBXML_XINCLUDE | LIBXML_NOERROR | LIBXML_NOWARNING);
							// information about the file, particularly row and column representation in XML
							$file_info = $sheet1_xml -> saveXML();
							if ( function_exists('simplexml_load_string') ) {
								$file_content = simplexml_load_string($file_content);
								$file_content = get_object_vars($file_content);
								$file_info = simplexml_load_string($file_info);
								$rows = get_object_vars($file_info -> sheetData);
								$rows = $rows['row'];
								$rows_total = count($rows);
								$result = array();
								foreach ($rows as $row) {
									$row = get_object_vars($row);
									$row_columns_total = count($row['c']);
									$row_data = array();
									foreach ($row['c'] as $cell) {
										$cell = get_object_vars($cell);
										// check if the array key is available in the main data array,
										// where we obtain the actual data: $file_content['si']
										// I've seen cases where there is data in the cell, but the value for $cell['v'] IS the actual data,
										// and NOT the corresponding ID for $file_content['si'].
										// also, I noticed if $cell['@attributes']['t'] is there, then the actual data is in $file_content['si'],
										// otherwise the data is $cell['v']
										if ( isset( $cell['@attributes']['t'] ) && isset( $file_content['si'][ (int)$cell['v'] ] ) ) {
											$cell_data = get_object_vars( $file_content['si'][ (int)$cell['v'] ] );
											$cell_data = $cell_data['t'];
										}
										else {
											// if we get here, the value of $cell['v'] should be the actual data,
											// and NOT the corresponding ID for $file_content['si']
											$cell_data = $cell['v'];
										}
										$row_data[] = $cell_data;
									}
									$result[] = $row_data;
								}
								if (!$result) {
									return array( "error" => _a("No data found") );
								}
								$header = current($result);
								$file_content = ac_array_csv($result, $header, $output = array());
							}
							else {
								return array( "error" => _a("PHP SimpleXML extension required") );
							}
						}
					}

				break;

				case "pptx" :

					// loop through all slide#.xml files
					$slide = 1;

					while ( ($index = $zip -> locateName("ppt/slides/slide" . $slide . ".xml")) !== false ) {

						$data = $zip -> getFromIndex($index);
						$xml = DOMDocument::loadXML($data, LIBXML_NOENT | LIBXML_XINCLUDE | LIBXML_NOERROR | LIBXML_NOWARNING);
						$file_content .= strip_tags($xml -> saveXML());

						$slide++;
					}

				break;
			}
		}

		$zip -> close();

		return $file_content;
	}
	else {
		return array( "error" => _a("PHP Zip extension required") );
	}
}

// grab just the text content from the file
function ac_file_upload_read($mimetype, $filename, $content, $ext, $return_format = "text") {

	$whitelist = array("htm", "html", "txt", "pdf", "doc", "odt", "docx", "xlsx", "pptx");
	if ( !in_array($ext, $whitelist) ) return "";

	if ($mimetype == "text/html") {

		// html

		$content = ac_str_strip_tags($content);
	}
	elseif ($mimetype == "application/pdf") {

		// pdf

		require_once ac_global_functions("pdf.php");

	  $searchstart = 'stream';
	  $searchend = 'endstream';
	  $pdfText = '';
	  $pos = 0;
	  $pos2 = 0;
	  $startpos = 0;

	  while ($pos !== false && $pos2 !== false) {

	    $pos = strpos($content, $searchstart, $startpos);
	    $pos2 = strpos($content, $searchend, $startpos + 1);

	    if ($pos !== false && $pos2 !== false){

	      if ($content[$pos] == 0x0d && $content[$pos + 1] == 0x0a) {
	      	$pos += 2;
	      }
	      else if ($content[$pos] == 0x0a) {
	      	$pos++;
	      }

	      if ($content[$pos2 - 2] == 0x0d && $content[$pos2 - 1] == 0x0a) {
	      	$pos2 -= 2;
	      }
	      else if ($content[$pos2 - 1] == 0x0a) {
	      	$pos2--;
	      }

	      $textsection = substr(
	        $content,
	        $pos + strlen($searchstart) + 2,
	        $pos2 - $pos - strlen($searchstart) - 1
	      );
	      $data = @gzuncompress($textsection);
	      $pdfText .= ac_pdf_extract_text($data);
	      $startpos = $pos2 + strlen($searchend) - 1;
	    }
	  }

	  $content = preg_replace('/(\s)+/', ' ', $pdfText);
	  $content = preg_replace('/\x91|\x92|\x96/', '', $content);
	}
	elseif ($mimetype == "application/msword") {

		// doc

		// script taken from: http://coding.derkeiler.com/Archive/PHP/php.general/2008-12/msg00213.html
		// only works with ".doc" files Word 97-03

		//$headers = substr($content, 0, 0xA00);
		$headers = substr($content, 0, 2560);

		// 1 = (ord(n)*1) ; Document has from 0 to 255 characters
		$n1 = ( ord($headers[0x21C]) - 1 );

		// 1 = ((ord(n)-8)*256) ; Document has from 256 to 63743 characters
		$n2 = ( ( ord($headers[0x21D]) - 8 ) * 256 );

		// 1 = ((ord(n)*256)*256) ; Document has from 63744 to 16775423 characters
		$n3 = ( ( ord($headers[0x21E]) * 256 ) * 256 );

		// (((ord(n)*256)*256)*256) ; Document has from 16775424 to 4294965504 characters
		$n4 = ( ( ( ord($headers[0x21F]) * 256 ) * 256 ) * 256 );

		// total length of text in the document
		$textLength = ($n1 + $n2 + $n3 + $n4);

		$content = substr($content, 2560, $textLength);
	}
	elseif ($mimetype == "application/octet-stream") {

		switch ($ext) {

			case "odt" :
			case "docx" :
			case "xlsx" :
			case "pptx" :

				$content = ac_file_zip_read_xml($filename, $ext, $return_format);

			break;

			/*
			case "xls" :
			case "ppt" :
			case "wps" :
			*/
		}
	}

	// we either receive the content back as a string, or an array that has a key for "error"
	if ( !is_array($content) ) {
		if ( strlen($content) > 65000 ) $content = substr($content, 0, 65000);
	}

	return $content;
}

// delete all files in folder X older than Z days (mask is a preg pattern for files)
function ac_file_delete_old($dir, $days, $mask = '') {
	$seconds = $days * 24 * 60 * 60;

	$files = ac_file_find($dir, $mask);
	foreach ( $files as $fname ) {
		$path = "$dir/$fname";
		//if ( $mask and !preg_match($mask, $path) ) continue;
		$mod_time = @filemtime($path);
		if ( !$mod_time ) continue;
		if ( file_exists($path) && time() - $mod_time > $seconds ) {
			@unlink($path);
		}
	}
}

?>
